import { spawn, execSync } from 'child_process';
import { join, dirname } from 'path';
import { fileURLToPath } from 'url';

const __dirname = dirname(fileURLToPath(import.meta.url));
const AGENT_ROOT = join(__dirname, '..');
const WORKSPACE_DIR = join(AGENT_ROOT, 'workspace');

export const name = 'claude';

export const modelMap = {
  fast: 'haiku',
  balanced: 'sonnet',
  powerful: 'opus'
};

export function isAvailable() {
  try {
    execSync(process.platform === 'win32' ? 'where claude' : 'which claude', { stdio: 'ignore' });
    return true;
  } catch {
    return false;
  }
}

/**
 * Spawn Claude CLI and return the result.
 *
 * @param {string} prompt
 * @param {object} opts
 * @param {string} opts.model
 * @param {number} opts.timeout - ms before killing
 * @param {boolean} opts.sandboxed - if true (default), run in workspace/ with hooks
 * @param {function} opts.onStream - callback(partialText) for streaming updates
 * @param {string[]} opts.extraArgs - additional CLI args
 */
export async function ask(prompt, { model = 'sonnet', timeout = 600_000, sandboxed = true, onStream, extraArgs = [] } = {}) {
  return new Promise((resolve, reject) => {
    const args = [
      '-p', prompt,
      '--model', model,
      '--output-format', 'stream-json',
      '--verbose',
      ...extraArgs
    ];

    const proc = spawn('claude', args, {
      cwd: sandboxed ? WORKSPACE_DIR : AGENT_ROOT,
      env: { ...process.env },
      stdio: ['pipe', 'pipe', 'pipe'],
      shell: process.platform === 'win32'
    });

    let fullResult = '';
    let accumulatedText = '';
    let buffer = '';
    let killed = false;

    const timer = setTimeout(() => {
      killed = true;
      proc.kill('SIGTERM');
      setTimeout(() => proc.kill('SIGKILL'), 5000);
    }, timeout);

    proc.stdout.on('data', (chunk) => {
      buffer += chunk.toString();
      const lines = buffer.split('\n');
      buffer = lines.pop();

      for (const line of lines) {
        if (!line.trim()) continue;
        try {
          const event = JSON.parse(line);

          if (event.type === 'assistant' && event.message?.content) {
            // Each assistant event contains text for that turn.
            // Multiple assistant events can occur (e.g. after tool use).
            // Accumulate all text and stream the full accumulated content.
            for (const block of event.message.content) {
              if (block.type === 'text' && block.text) {
                accumulatedText += block.text;
              }
            }
            if (onStream) onStream(accumulatedText);
          } else if (event.type === 'result') {
            fullResult = event.result || '';
          }
        } catch {
          // Ignore malformed lines
        }
      }
    });

    let stderr = '';
    proc.stderr.on('data', (chunk) => { stderr += chunk.toString(); });

    proc.on('close', (code) => {
      clearTimeout(timer);
      if (killed) {
        reject(new Error(`Claude CLI timed out after ${timeout}ms`));
      } else if (code !== 0 && !fullResult) {
        reject(new Error(`Claude CLI exited with code ${code}\n${stderr}`));
      } else {
        resolve(fullResult.trim());
      }
    });

    proc.on('error', (err) => {
      clearTimeout(timer);
      reject(new Error(`Claude CLI spawn error: ${err.message}`));
    });

    proc.stdin.end();
  });
}

/**
 * Spawn Claude in interactive mode (for meta-develop).
 * Returns the spawned child process.
 */
export function spawnInteractive({ cwd, model = 'claude-opus-4-6' } = {}) {
  return spawn('claude', [
    '--model', model,
    '--dangerously-skip-permissions',
    '--output-format', 'stream-json',
    '--verbose'
  ], {
    cwd: cwd || WORKSPACE_DIR,
    env: { ...process.env },
    stdio: ['pipe', 'pipe', 'pipe'],
    shell: process.platform === 'win32'
  });
}
