import { spawn, execSync } from 'child_process';
import { join, dirname } from 'path';
import { fileURLToPath } from 'url';

const __dirname = dirname(fileURLToPath(import.meta.url));
const AGENT_ROOT = join(__dirname, '..');
const WORKSPACE_DIR = join(AGENT_ROOT, 'workspace');

export const name = 'codex';

export const modelMap = {
  fast: 'gpt-5-codex-mini',
  balanced: 'gpt-5.2-codex',
  powerful: 'gpt-5.3-codex'
};

export function isAvailable() {
  try {
    execSync(process.platform === 'win32' ? 'where codex' : 'which codex', { stdio: 'ignore' });
    return true;
  } catch {
    return false;
  }
}

/**
 * Run Codex CLI in non-interactive (exec) mode and return the result.
 *
 * @param {string} prompt
 * @param {object} opts
 * @param {string} opts.model
 * @param {number} opts.timeout - ms before killing
 * @param {boolean} opts.sandboxed - if true (default), run in workspace/
 * @param {function} opts.onStream - callback(partialText) for streaming updates
 */
export async function ask(prompt, { model = 'gpt-5.2-codex', timeout = 600_000, sandboxed = true, onStream } = {}) {
  return new Promise((resolve, reject) => {
    // Pass prompt via stdin (using '-') to avoid shell escaping issues on Windows
    const args = [
      'exec',
      '--json',
      '--model', model,
      '-'
    ];

    const proc = spawn('codex', args, {
      cwd: sandboxed ? WORKSPACE_DIR : AGENT_ROOT,
      env: { ...process.env },
      stdio: ['pipe', 'pipe', 'pipe'],
      shell: process.platform === 'win32'
    });

    let fullResult = '';
    let accumulatedText = '';
    let buffer = '';
    let killed = false;

    const timer = setTimeout(() => {
      killed = true;
      proc.kill('SIGTERM');
      setTimeout(() => proc.kill('SIGKILL'), 5000);
    }, timeout);

    proc.stdout.on('data', (chunk) => {
      buffer += chunk.toString();
      const lines = buffer.split('\n');
      buffer = lines.pop();

      for (const line of lines) {
        if (!line.trim()) continue;
        try {
          const raw = JSON.parse(line);

          // Codex wraps events in {id, msg} envelope
          const event = raw.msg || raw;

          if (event.type === 'agent_message' && event.message) {
            accumulatedText += event.message;
            if (onStream) onStream(accumulatedText);
          } else if (event.type === 'item.completed' && event.item?.content) {
            for (const block of event.item.content) {
              if (block.type === 'output_text' && block.text) {
                accumulatedText += block.text;
                if (onStream) onStream(accumulatedText);
              }
            }
          } else if (event.type === 'message.completed' && event.message?.content) {
            for (const block of event.message.content) {
              if (block.type === 'output_text' && block.text) {
                accumulatedText += block.text;
                if (onStream) onStream(accumulatedText);
              }
            }
          }
        } catch {
          // Non-JSON line — may be plain text result
          if (line.trim()) {
            accumulatedText += line;
            if (onStream) onStream(accumulatedText);
          }
        }
      }
    });

    let stderr = '';
    proc.stderr.on('data', (chunk) => { stderr += chunk.toString(); });

    proc.on('close', (code) => {
      clearTimeout(timer);

      // Process any remaining buffer
      if (buffer.trim()) {
        try {
          const raw = JSON.parse(buffer);
          const event = raw.msg || raw;
          if (event.type === 'agent_message' && event.message) {
            accumulatedText += event.message;
          } else if (event.type === 'item.completed' && event.item?.content) {
            for (const block of event.item.content) {
              if (block.type === 'output_text' && block.text) {
                accumulatedText += block.text;
              }
            }
          }
        } catch {
          if (buffer.trim()) accumulatedText += buffer.trim();
        }
      }

      fullResult = accumulatedText.trim();

      if (killed) {
        reject(new Error(`Codex CLI timed out after ${timeout}ms`));
      } else if (code !== 0 && !fullResult) {
        reject(new Error(`Codex CLI exited with code ${code}\n${stderr}`));
      } else {
        resolve(fullResult);
      }
    });

    proc.on('error', (err) => {
      clearTimeout(timer);
      reject(new Error(`Codex CLI spawn error: ${err.message}`));
    });

    // Write prompt to stdin and close
    proc.stdin.write(prompt);
    proc.stdin.end();
  });
}

/**
 * Spawn Codex in interactive mode (for meta-develop).
 * Returns the spawned child process.
 */
export function spawnInteractive({ cwd, model = 'gpt-5.2-codex' } = {}) {
  return spawn('codex', [
    '--model', model,
    '--full-auto'
  ], {
    cwd: cwd || WORKSPACE_DIR,
    env: { ...process.env },
    stdio: ['pipe', 'pipe', 'pipe'],
    shell: process.platform === 'win32'
  });
}
