import { spawn, execSync } from 'child_process';
import { join, dirname } from 'path';
import { fileURLToPath } from 'url';

const __dirname = dirname(fileURLToPath(import.meta.url));
const AGENT_ROOT = join(__dirname, '..');
const WORKSPACE_DIR = join(AGENT_ROOT, 'workspace');

export const name = 'gemini';

export const modelMap = {
  fast: 'gemini-2.5-flash',
  balanced: 'gemini-2.5-pro',
  powerful: 'gemini-2.5-pro'
};

export function isAvailable() {
  try {
    execSync(process.platform === 'win32' ? 'where gemini' : 'which gemini', { stdio: 'ignore' });
    return true;
  } catch {
    return false;
  }
}

/**
 * Run Gemini CLI in headless mode and return the result.
 *
 * @param {string} prompt
 * @param {object} opts
 * @param {string} opts.model
 * @param {number} opts.timeout - ms before killing
 * @param {boolean} opts.sandboxed - if true (default), run in workspace/
 * @param {function} opts.onStream - callback(partialText) for streaming updates
 */
export async function ask(prompt, { model = 'gemini-2.5-pro', timeout = 600_000, sandboxed = true, onStream } = {}) {
  return new Promise((resolve, reject) => {
    const args = [
      '--model', model,
      '--output-format', 'stream-json',
      '-p', prompt
    ];

    const proc = spawn('gemini', args, {
      cwd: sandboxed ? WORKSPACE_DIR : AGENT_ROOT,
      env: { ...process.env },
      stdio: ['pipe', 'pipe', 'pipe'],
      shell: process.platform === 'win32'
    });

    let accumulatedText = '';
    let buffer = '';
    let killed = false;

    const timer = setTimeout(() => {
      killed = true;
      proc.kill('SIGTERM');
      setTimeout(() => proc.kill('SIGKILL'), 5000);
    }, timeout);

    proc.stdout.on('data', (chunk) => {
      buffer += chunk.toString();
      const lines = buffer.split('\n');
      buffer = lines.pop();

      for (const line of lines) {
        if (!line.trim()) continue;
        try {
          const event = JSON.parse(line);

          // Gemini stream-json: look for text content in various event shapes
          if (event.type === 'result' && event.result) {
            accumulatedText = event.result;
            if (onStream) onStream(accumulatedText);
          } else if (event.type === 'assistant' && event.message?.content) {
            for (const block of event.message.content) {
              if (block.type === 'text' && block.text) {
                accumulatedText += block.text;
              }
            }
            if (onStream) onStream(accumulatedText);
          } else if (event.text) {
            // Simple text event
            accumulatedText += event.text;
            if (onStream) onStream(accumulatedText);
          } else if (event.content) {
            // Direct content field
            accumulatedText = typeof event.content === 'string' ? event.content : accumulatedText;
            if (onStream) onStream(accumulatedText);
          }
        } catch {
          // Non-JSON line — treat as plain text output
          if (line.trim()) {
            accumulatedText += line;
            if (onStream) onStream(accumulatedText);
          }
        }
      }
    });

    let stderr = '';
    proc.stderr.on('data', (chunk) => { stderr += chunk.toString(); });

    proc.on('close', (code) => {
      clearTimeout(timer);

      // Process remaining buffer
      if (buffer.trim()) {
        try {
          const event = JSON.parse(buffer);
          if (event.type === 'result' && event.result) {
            accumulatedText = event.result;
          } else if (event.text) {
            accumulatedText += event.text;
          }
        } catch {
          if (buffer.trim()) accumulatedText += buffer.trim();
        }
      }

      const result = accumulatedText.trim();

      if (killed) {
        reject(new Error(`Gemini CLI timed out after ${timeout}ms`));
      } else if (code !== 0 && !result) {
        reject(new Error(`Gemini CLI exited with code ${code}\n${stderr}`));
      } else {
        resolve(result);
      }
    });

    proc.on('error', (err) => {
      clearTimeout(timer);
      reject(new Error(`Gemini CLI spawn error: ${err.message}`));
    });

    proc.stdin.end();
  });
}

/**
 * Spawn Gemini in interactive mode (for meta-develop).
 * Returns the spawned child process.
 */
export function spawnInteractive({ cwd, model = 'gemini-2.5-pro' } = {}) {
  return spawn('gemini', [
    '--model', model,
    '--sandbox'
  ], {
    cwd: cwd || WORKSPACE_DIR,
    env: { ...process.env },
    stdio: ['pipe', 'pipe', 'pipe'],
    shell: process.platform === 'win32'
  });
}
