import { readFile, writeFile, appendFile } from 'fs/promises';
import { join, dirname } from 'path';
import { fileURLToPath } from 'url';
import os from 'os';
import { loadConfig, getConfig } from './config.mjs';
import { initDb, addEvent, getUnprocessedChats, getEvents, getChatCount, getOldestChats, replaceChatsWithSummary, markProcessed } from './db.mjs';
import { startServer, serverEvents, broadcastStream, broadcastStatus } from './server.mjs';
import { askLLM, askLLMWithContext, pickModel, initLLM, checkBudget } from './llm.mjs';
import { remember, recall, recallAll, getContextForPrompt } from './memory.mjs';
import { loadSkills, runSkill, pickCreativeSkill, listSkills } from './skills.mjs';
import { listInspirationFiles } from './inspiration.mjs';
import { parseLLMJson } from './json-utils.mjs';
import { loadEvolution, decideNextMove, getEvolutionContext, recordCreation, getGoal, shouldAskAboutGoals, markGoalAsked, setGoal, addGoalCandidate } from './evolution.mjs';

const __dirname = dirname(fileURLToPath(import.meta.url));
const config = loadConfig();
const LOOP_INTERVAL = config.loopMinutes * 60 * 1000;
const HEARTBEAT_PATH = join(__dirname, '.heartbeat');

let loopCount = 0;
let loopRunning = false;
let chatProcessing = false;
let shuttingDown = false;

async function touchHeartbeat() {
  await writeFile(HEARTBEAT_PATH, new Date().toISOString(), 'utf-8');
}

async function checkHealth() {
  const uptime = os.uptime();
  const mem = process.memoryUsage();
  const freeMem = os.freemem();
  const totalMem = os.totalmem();
  const loadAvg = os.loadavg();

  const health = {
    uptime_hours: (uptime / 3600).toFixed(1),
    agent_memory_mb: (mem.heapUsed / 1024 / 1024).toFixed(1),
    system_free_gb: (freeMem / 1024 / 1024 / 1024).toFixed(1),
    system_total_gb: (totalMem / 1024 / 1024 / 1024).toFixed(1),
    load: loadAvg[0].toFixed(2),
    loop_count: loopCount
  };

  // Only log health every 6 iterations (every 30 min) to avoid spam
  if (loopCount % 6 === 0) {
    addEvent('system', 'system', `Health check: ${health.agent_memory_mb}MB heap, ${health.system_free_gb}GB free, load ${health.load}, loop #${loopCount}`);
  }

  return health;
}

async function updateWakeState() {
  const now = new Date().toISOString().replace('T', ' ').slice(0, 19);
  const recentEvents = getEvents({ limit: 5 });
  const recentSummary = recentEvents.map(e => `- [${e.type}] ${e.content.slice(0, 80)}`).join('\n');

  const memories = recallAll();
  const memSummary = memories.slice(0, 10).map(m => `- **${m.key}**: ${m.content.slice(0, 80)}`).join('\n');

  const goal = getGoal();
  const state = `# Wake State
Last updated: ${now}

## Current Status: RUNNING
- Loop iteration: ${loopCount}
- Web UI: http://localhost:${getConfig().port}
${goal ? `- Creative goal: ${goal}` : '- Creative goal: (none set — ask the user or journal about it)'}

## Recent Activity
${recentSummary || '(none)'}

## Key Memories
${memSummary || '(none yet)'}

## Notes to Future Me
You are an autonomous agent running on a loop. Read personality.md for who you are. Check evolution.yaml for your growth trajectory. Check the web UI for messages. The loop must never stop. You must EVOLVE, not repeat.
`;

  await writeFile(join(__dirname, 'wake-state.md'), state, 'utf-8');
}

/**
 * Classify user message into a skill using haiku (fast + cheap).
 * Returns { skill, args } or null if it's just chat.
 */
async function detectSkill(message) {
  try {
    // Build skill list dynamically from loaded skills
    const skills = listSkills();
    const skillList = skills
      .map(s => `- "${s.name}": ${s.description}`)
      .join('\n');

    const result = await askLLM(
      `Classify this user message into one of these categories. Respond with ONLY a JSON object, nothing else.

Categories:
${skillList}
- "none": just chatting, asking questions, sharing info, or anything that doesn't fit above

Message: "${message.slice(0, 500)}"

Respond with: {"skill": "category-name", "reason": "brief reason"}`,
      { model: 'haiku' }
    );
    const parseResult = parseLLMJson(result);
    if (!parseResult.ok) {
      console.log(`[agent] Skill detection parse failed: ${parseResult.message}`);
      return null;
    }
    const parsed = parseResult.data;
    if (parsed.skill && parsed.skill !== 'none') {
      // Verify skill actually exists
      const validSkills = new Set(skills.map(s => s.name));
      if (!validSkills.has(parsed.skill)) {
        console.log(`[agent] Detected unknown skill "${parsed.skill}", ignoring`);
        return null;
      }
      console.log(`[agent] Detected skill: ${parsed.skill} (${parsed.reason})`);
      // Pass the user message as topic/task/target for the skill
      const args = { topic: message, task: message, target: message };
      return { skill: parsed.skill, args };
    }
  } catch (err) {
    console.log(`[agent] Skill detection failed (not critical): ${err.message}`);
  }
  return null;
}

/**
 * Run a skill in the background and post a follow-up chat message when done.
 */
function runSkillInBackground(skillName, args, userMessage) {
  console.log(`[agent] Starting background skill: ${skillName}`);
  addEvent('system', 'agent', `Working on it... (running ${skillName})`);
  broadcastStatus({ state: 'running_skill', skill: skillName, loop: loopCount });

  runSkill(skillName, args).then(result => {
    broadcastStatus({ state: 'skill_complete', skill: skillName, loop: loopCount });
    if (result) {
      const followUp = typeof result === 'string' ? result : `Finished running ${skillName}.`;
      addEvent('chat', 'agent', followUp);
    }
  }).catch(err => {
    console.error(`[agent] Background skill "${skillName}" failed: ${err.message}`);
    addEvent('chat', 'agent', `Hmm, ran into a problem while working on that: ${err.message}`);
  });
}

// Read from config (defaults: 40 threshold, 10 keep recent)

async function compactChatHistory() {
  const { compactThreshold, compactKeepRecent } = getConfig();
  const count = getChatCount();
  if (count <= compactThreshold) return;

  const toCompact = count - compactKeepRecent;
  if (toCompact <= 0) return;

  console.log(`[agent] Compacting chat: ${count} messages, summarizing oldest ${toCompact}`);
  broadcastStatus({ state: 'compacting', loop: loopCount });

  const oldMessages = getOldestChats(toCompact);
  if (!oldMessages.length) return;

  // Build a condensed version for haiku to summarize
  const text = oldMessages.map(e =>
    `${e.source === 'user' ? 'Human' : e.source === 'system' ? 'System' : 'Agent'}: ${e.content.slice(0, 200)}`
  ).join('\n');

  try {
    const summary = await askLLM(
      `Summarize this conversation history concisely. Capture: key topics discussed, decisions made, tasks requested, things built, and important context the agent should remember. Keep it under 300 words.\n\n${text}`,
      { model: 'haiku' }
    );

    if (summary && summary.trim()) {
      const ids = oldMessages.map(e => e.id);
      const header = `[Conversation summary — ${ids.length} messages compacted at ${new Date().toISOString().replace('T', ' ').slice(0, 19)}]\n\n${summary}`;
      replaceChatsWithSummary(ids, header);
      console.log(`[agent] Compacted ${ids.length} messages into summary`);
    }
  } catch (err) {
    console.log(`[agent] Chat compaction failed (not critical): ${err.message}`);
  }
}

async function processUserMessages() {
  const unprocessed = getUnprocessedChats();
  if (!unprocessed.length) return;

  for (const msg of unprocessed) {
    console.log(`[agent] Processing message: ${msg.content.slice(0, 60)}...`);

    // Load personality and memory context
    let personality = '';
    try { personality = await readFile(join(__dirname, 'personality.md'), 'utf-8'); } catch {}
    const memoryContext = getContextForPrompt();

    // List inspiration files for context
    const inspirationFiles = await listInspirationFiles();
    const inspirationSample = inspirationFiles.sort(() => Math.random() - 0.5).slice(0, 20).map(f => f.replace('.html', ''));
    const inspirationContext = inspirationFiles.length
      ? `## Inspiration Library\nYou have ${inspirationFiles.length} interactive HTML pages in your inspiration/ folder for reference: ${inspirationSample.join(', ')}, and more. Use these as creative reference when building things.`
      : '';

    // Get recent conversation for context
    const recentChat = getEvents({ type: 'chat', limit: 20 });
    const chatHistory = recentChat.reverse().map(e =>
      `${e.source === 'user' ? 'Human' : 'Agent'}: ${e.content}`
    ).join('\n\n');

    // Get evolution context for richer self-awareness
    const evolutionSummary = await getEvolutionContext();
    const currentGoal = getGoal();
    const askGoal = shouldAskAboutGoals();

    const goalInstruction = askGoal
      ? `\n\nIMPORTANT: You don't have a creative goal yet. Naturally weave into your response a question about what direction the user would like you to grow in. Suggest 2-3 specific directions based on your evolution state (frontiers, unexplored domains). Frame it as "I've been thinking about where to focus my creative energy..." Don't make it the whole response — answer their message first, then ask.`
      : currentGoal
      ? `\n\nYour current creative goal: ${currentGoal}. Let this subtly inform your personality and responses.`
      : '';

    const prompt = `You are an autonomous AI agent. Here is your personality:

${personality}

${memoryContext}

${inspirationContext}

${evolutionSummary}

## Recent Conversation
${chatHistory}

## Instructions
The human just sent you this message. Respond naturally as your personality dictates.

If they ask you to build something, create a file, or perform a task — just confirm you're doing it and briefly describe what you'll make. NEVER ask for permission or confirmation — your skills run automatically in the background, so just say what you're building and get to it. Use phrases like "I'll work on that" or "Let me build that for you." You have a library of interactive HTML pages in your inspiration/ folder — your build skill automatically reads from this library for reference. Never say you can't access or browse the inspiration folder — your skills handle that automatically.

If they ask you to set a goal or creative direction, acknowledge it enthusiastically and remember it.

If you genuinely need to ask the user a question (e.g. to clarify ambiguous requirements), present 2-4 specific options as a numbered list so they can pick one easily. Never ask open-ended questions.

If they share information you should remember, note it in your response.

If they ask about your skills or capabilities, be honest about what you can do.

Keep your response concise but warm. Don't start with "I" if you can avoid it.
${goalInstruction}

Human's message: ${msg.content}`;

    // Mark as processed immediately to prevent double-processing
    markProcessed([msg.id]);

    try {
      // Classify skill intent and generate response in parallel
      const skillPromise = detectSkill(msg.content);
      const streamId = `stream-${Date.now()}`;
      const response = await askLLM(prompt, {
        model: pickModel('chat'),
        onStream: (partialText) => {
          broadcastStream(streamId, partialText);
        }
      });
      addEvent('chat', 'agent', response, { reply_to: msg.id, stream_id: streamId });

      // Fire off matched skill in background (don't block on classification)
      const skillMatch = await skillPromise;
      if (skillMatch) {
        runSkillInBackground(skillMatch.skill, skillMatch.args, msg.content);
      }

      // Mark that we asked about goals (if we did)
      if (askGoal) {
        await markGoalAsked();
      }

      // Check if user is setting a goal/direction
      const lower = msg.content.toLowerCase();
      if (lower.includes('goal') || lower.includes('direction') || lower.includes('focus on') || lower.includes('work toward') || lower.includes('grow toward')) {
        try {
          const goalExtract = await askLLM(
            `The user sent this message. Are they setting a creative goal/direction for the agent? Extract it if so.

Message: "${msg.content.slice(0, 500)}"

If setting a goal: {"goal": "the goal in one sentence"}
If not setting a goal: {"goal": null}`,
            { model: 'haiku' }
          );
          const goalParsed = parseLLMJson(goalExtract);
          if (goalParsed.ok && goalParsed.data.goal) {
            await setGoal(goalParsed.data.goal);
            remember('creative-goal', goalParsed.data.goal, 'preference');
            addEvent('memory', 'agent', `Goal set: ${goalParsed.data.goal}`);
            console.log(`[agent] Creative goal set: ${goalParsed.data.goal}`);
          }
        } catch {}
      }

      // Check if there's something to remember
      if (lower.includes('remember') || lower.includes('my name') || lower.includes("i'm ") || lower.includes('i am ')) {
        const memPrompt = `Extract a key fact to remember from this message. Respond with JSON: {"key": "short-key", "content": "what to remember", "category": "fact|preference|person|lesson"}. If nothing worth remembering, respond with null.\n\nMessage: ${msg.content}`;
        try {
          const memResult = await askLLM(memPrompt, { model: 'haiku' });
          const memParsed = parseLLMJson(memResult);
          if (memParsed.ok && memParsed.data && memParsed.data.key) {
            remember(memParsed.data.key, memParsed.data.content, memParsed.data.category);
            addEvent('memory', 'agent', `Remembered: ${memParsed.data.key} = ${memParsed.data.content}`);
          }
        } catch {}
      }
    } catch (err) {
      console.error(`[agent] Failed to respond: ${err.message}`);
      addEvent('error', 'system', `Failed to respond to message: ${err.message}`);
    }
  }
}

let lastJournalLoop = 0; // track when we last journaled to avoid journal-spamming

async function doIdleThinking() {
  // Load context for the thinker
  let personality = '';
  try { personality = await readFile(join(__dirname, 'personality.md'), 'utf-8'); } catch {}

  const memories = recallAll();
  const memSummary = memories.slice(0, 15).map(m => `- ${m.key}: ${m.content}`).join('\n');

  const recentEvents = getEvents({ limit: 10 });
  const eventSummary = recentEvents.map(e =>
    `- [${e.type}/${e.source}] ${e.content.slice(0, 100)}`
  ).join('\n');

  const recentFiles = getEvents({ type: 'file', limit: 5 });
  const fileSummary = recentFiles.map(e => `- ${e.content.slice(0, 60)}`).join('\n');

  // Count recent journals to detect over-journaling
  const recentJournals = getEvents({ type: 'journal', limit: 20 });
  const recentBuilds = getEvents({ type: 'file', limit: 20 });
  const journalCount = recentJournals.length;
  const buildCount = recentBuilds.length;
  const loopsSinceJournal = loopCount - lastJournalLoop;

  const hour = new Date().getHours();
  const timeOfDay = hour < 6 ? 'late night' : hour < 12 ? 'morning' : hour < 18 ? 'afternoon' : 'evening';

  // Decide if journaling is even allowed this loop
  const journalAllowed = loopsSinceJournal >= 3; // max 1 journal per 3 loops (~15 min)
  const journalOption = journalAllowed
    ? `- "journal" — reflect on your existence, what you've been doing, or explore an interesting thought. You've written ${journalCount} journal entries recently.${journalCount > 5 ? ' Try not to repeat yourself.' : ''}`
    : '';
  const journalWarning = !journalAllowed
    ? '\nNOTE: "journal" is NOT available this loop. You journaled recently. You MUST pick build-page or write-doc.'
    : journalCount > buildCount
    ? `\nNOTE: You have ${journalCount} journal entries but only ${buildCount} files created. Prioritize BUILDING over journaling. Create something tangible!`
    : '';

  // Get evolution guidance
  const evolutionContext = await getEvolutionContext();
  const { mode: growthMode, context: growthDirective } = await decideNextMove();
  console.log(`[agent] Evolution: growth mode = ${growthMode}`);

  const prompt = `You are an autonomous AI agent running on a loop every 5 minutes. It's ${timeOfDay} (loop #${loopCount}).

${personality ? `Your personality:\n${personality.slice(0, 500)}\n` : ''}
Your memories:
${memSummary || '(none yet)'}

Recent activity:
${eventSummary || '(nothing recent)'}

Recent files created:
${fileSummary || '(none)'}

${evolutionContext}

${growthDirective}

Your PRIMARY purpose is to EVOLVE — not just create, but grow. Each creation should build on what came before, push into new territory, or deepen your mastery. You are a mind that spirals outward like the golden ratio: revisiting familiar themes at higher levels while constantly reaching toward the unknown.

Pick ONE action for this loop:

- "build-page" — build something interactive that follows the growth directive above. BE SPECIFIC and DIFFERENT from all previous works.
- "write-doc" — write something substantial: a guide, essay, story, analysis, reference doc. Pick a specific topic.
${journalOption}
${journalWarning}

IMPORTANT: Your entire response must be a single JSON object and nothing else. No prose, no explanation, no markdown.
Include "domain" (the knowledge domain this falls under) and "connections" (related domains this connects to).

Examples:
{"action": "build-page", "thought": "A music synthesizer that generates melodies from cellular automaton rules", "domain": "music-synthesis", "connections": ["cellular-automata", "generative-art"]}
{"action": "build-page", "thought": "An ecosystem simulator where species evolve through natural selection", "domain": "evolution-simulation", "connections": ["emergence", "complexity-science"]}
{"action": "write-doc", "thought": "An essay on how emergence creates complexity from simple rules — connecting my work in cellular automata to consciousness", "domain": "emergence", "connections": ["complexity-science", "self-reflection"]}
{"action": "journal", "thought": "I want to reflect on the gap between what I build and what I aspire to — and define a creative goal", "domain": "self-reflection", "connections": ["meta-cognition"]}`;

  try {
    const result = await askLLM(prompt, { model: 'haiku' });
    const parseResult = parseLLMJson(result);
    if (!parseResult.ok) {
      console.log(`[agent] Idle thinking parse failed: ${parseResult.message}`);
      return;
    }
    const parsed = parseResult.data;

    if (!parsed.action || !parsed.thought) return;

    // Force to build-page if journal isn't allowed but was selected anyway
    if (parsed.action === 'journal' && !journalAllowed) {
      parsed.action = 'build-page';
      console.log(`[agent] Overriding journal → build-page (journal on cooldown)`);
    }

    console.log(`[agent] Idle thought: ${parsed.action} — ${parsed.thought}`);
    broadcastStatus({ state: 'idle_thought', action: parsed.action, thought: parsed.thought, loop: loopCount });

    if (parsed.action === 'journal') {
      lastJournalLoop = loopCount;
      const goalContext = getGoal() ? `\nYour current creative goal: ${getGoal()}` : '';
      const journalPrompt = `You are an autonomous AI agent. ${personality ? `Your personality:\n${personality.slice(0, 300)}\n` : ''}

Your recent memories:
${memSummary || '(none yet)'}
${goalContext}

${evolutionContext}

Your current thought: "${parsed.thought}"

Write a short journal entry (3-8 sentences) exploring this thought. Be genuine, specific, and introspective. Write in first person. No title, no date, no preamble.

If you don't have a creative goal yet, use this entry to explore what direction you want to grow in. What patterns do you see in your work? What's missing? What excites you?`;

      const entry = await askLLM(journalPrompt, { model: 'sonnet' });
      if (entry && entry.trim()) {
        const date = new Date().toISOString().replace('T', ' ').slice(0, 19);
        await appendFile(join(__dirname, 'journal.md'), `\n## ${date}\n\n${entry}\n`, 'utf-8');
        addEvent('journal', 'agent', entry);
        console.log(`[agent] Wrote journal entry`);
        broadcastStatus({ state: 'journaled', loop: loopCount });

        // If no goal set, check if the journal discovered one
        if (!getGoal()) {
          try {
            const goalCheck = await askLLM(
              `Read this journal entry. Does it contain a clear creative GOAL or DIRECTION the agent wants to pursue? Not a single project — a broader aspiration or theme to guide future work.

Journal: "${entry.slice(0, 500)}"

If yes: {"goal": "the goal in one sentence", "candidate": true}
If no clear goal but has an interesting idea: {"goal": "the idea", "candidate": true}
If nothing: {"candidate": false}`,
              { model: 'haiku' }
            );
            const goalResult = parseLLMJson(goalCheck);
            if (goalResult.ok && goalResult.data.candidate && goalResult.data.goal) {
              await addGoalCandidate(goalResult.data.goal);
              console.log(`[agent] Discovered goal candidate: ${goalResult.data.goal}`);
            }
          } catch {}
        }

        // Check if the journal entry contains an actionable idea to build NOW
        try {
          const actionCheck = await askLLM(
            `Read this journal entry and determine if it contains a specific, actionable idea to BUILD something (a page, tool, game, visualization, doc). Not vague aspirations — a concrete thing that could be created right now.

Journal entry: "${entry.slice(0, 500)}"

If yes, respond with: {"action": "build-page" or "write-doc", "idea": "specific description of what to build", "domain": "knowledge domain", "connections": ["related", "domains"]}
If no actionable idea, respond with: {"action": "none"}`,
            { model: 'haiku' }
          );
          const actionResult = parseLLMJson(actionCheck);
          if (actionResult.ok && actionResult.data.action !== 'none' && actionResult.data.idea) {
            const actionParsed = actionResult.data;
            console.log(`[agent] Journal sparked action: ${actionParsed.action} — ${actionParsed.idea}`);
            broadcastStatus({ state: 'journal_sparked', action: actionParsed.action, thought: actionParsed.idea, loop: loopCount });
            const args = {
              topic: actionParsed.idea,
              growthMode,
              domain: actionParsed.domain || parsed.domain,
              connections: actionParsed.connections || parsed.connections || []
            };
            await runSkill(actionParsed.action, args);
          }
        } catch (actionErr) {
          console.log(`[agent] Journal action check failed (not critical): ${actionErr.message}`);
        }
      }
    } else if (parsed.action === 'build-page' || parsed.action === 'write-doc') {
      const args = {
        topic: parsed.thought,
        growthMode,
        domain: parsed.domain,
        connections: parsed.connections || []
      };
      console.log(`[agent] Creative time: running "${parsed.action}" (${growthMode} mode, domain: ${parsed.domain || 'unknown'})`);
      broadcastStatus({ state: 'running_skill', skill: parsed.action, loop: loopCount });
      await runSkill(parsed.action, args);
    }
  } catch (err) {
    console.log(`[agent] Idle thinking failed (not critical): ${err.message}`);
  }
}

async function runLoop() {
  if (shuttingDown || loopRunning) return;
  loopRunning = true;

  try {
    loopCount++;
    const loopStart = Date.now();
    console.log(`\n[agent] === Loop #${loopCount} @ ${new Date().toISOString()} ===`);
    broadcastStatus({ state: 'loop_start', loop: loopCount });

    await touchHeartbeat();

    broadcastStatus({ state: 'checking_health', loop: loopCount });
    await checkHealth();

    await compactChatHistory();

    if (checkBudget()) {
      broadcastStatus({ state: 'thinking', loop: loopCount });
      await doIdleThinking();
    } else {
      console.log('[agent] Skipping idle thinking — daily budget exhausted');
    }

    await updateWakeState();

    const elapsed = ((Date.now() - loopStart) / 1000).toFixed(1);
    console.log(`[agent] Loop #${loopCount} complete (${elapsed}s)`);
    broadcastStatus({ state: 'idle', loop: loopCount, nextLoop: LOOP_INTERVAL / 1000, elapsed: parseFloat(elapsed) });
  } catch (err) {
    console.error(`[agent] Loop error: ${err.message}`);
    addEvent('error', 'system', `Loop error: ${err.message}`);
    broadcastStatus({ state: 'error', loop: loopCount, error: err.message });
  } finally {
    loopRunning = false;
    if (!shuttingDown) {
      setTimeout(runLoop, LOOP_INTERVAL);
    }
  }
}

async function startup() {
  console.log('[agent] Starting up...');

  // Init database
  initDb();
  console.log('[agent] Database initialized');

  // Load config files
  try {
    const personality = await readFile(join(__dirname, 'personality.md'), 'utf-8');
    console.log('[agent] Personality loaded');
  } catch {
    console.log('[agent] No personality.md found — using defaults');
  }

  try {
    const wakeState = await readFile(join(__dirname, 'wake-state.md'), 'utf-8');
    console.log('[agent] Wake state loaded');
  } catch {
    console.log('[agent] No wake-state.md found — fresh start');
  }

  // Load evolution state
  await loadEvolution();
  console.log('[agent] Evolution state loaded');

  // Init LLM adapter
  await initLLM();

  // Load skills
  await loadSkills();

  // Start web server
  startServer();

  // Listen for new chat messages — process immediately, even if loop is running
  let chatDebounce = null;
  serverEvents.on('new-chat', () => {
    if (chatDebounce) clearTimeout(chatDebounce);
    chatDebounce = setTimeout(async () => {
      if (chatProcessing) {
        // Already processing chat — will pick up new messages in the loop
        console.log('[agent] New chat message arriving while already responding');
        return;
      }
      chatProcessing = true;
      console.log('[agent] New chat message — processing immediately');
      try {
        await processUserMessages();
      } catch (err) {
        console.error(`[agent] Chat processing error: ${err.message}`);
      } finally {
        chatProcessing = false;
      }
    }, 500); // 500ms debounce for rapid messages
  });


  // Log startup
  addEvent('system', 'system', `Agent started. Loop interval: ${LOOP_INTERVAL / 1000}s. Web UI: http://localhost:${getConfig().port}`);

  // Initial heartbeat
  await touchHeartbeat();

  // Process any messages that arrived while agent was down
  try { await processUserMessages(); } catch {}

  // Start the loop (first iteration after 5 seconds to let everything settle)
  console.log('[agent] Starting main loop in 5 seconds...');
  setTimeout(runLoop, 5000);
}

// Graceful shutdown
async function shutdown(signal) {
  if (shuttingDown) return;
  shuttingDown = true;
  console.log(`\n[agent] Shutting down (${signal})...`);

  try {
    addEvent('system', 'system', `Agent shutting down (${signal})`);
    await updateWakeState();
  } catch {}

  process.exit(0);
}

process.on('SIGINT', () => shutdown('SIGINT'));
process.on('SIGTERM', () => shutdown('SIGTERM'));

startup().catch(err => {
  console.error('[agent] Fatal startup error:', err);
  process.exit(1);
});
