import { readFileSync, writeFileSync, existsSync } from 'fs';
import { join, dirname } from 'path';
import { fileURLToPath } from 'url';
import { randomBytes } from 'crypto';

const __dirname = dirname(fileURLToPath(import.meta.url));
const CONFIG_PATH = join(__dirname, 'config.json');

const DEFAULTS = {
  cli: 'claude',
  port: 4000,
  loopMinutes: 5,
  compactThreshold: 40,
  compactKeepRecent: 10,
  agentName: 'agent',
  models: {
    fast: 'haiku',
    balanced: 'sonnet',
    powerful: 'opus'
  },
  dailyBudget: null,
  authToken: null
};

let _config = null;

export function loadConfig() {
  if (existsSync(CONFIG_PATH)) {
    try {
      const raw = readFileSync(CONFIG_PATH, 'utf-8');
      const userConfig = JSON.parse(raw);
      _config = { ...DEFAULTS, ...userConfig, models: { ...DEFAULTS.models, ...(userConfig.models || {}) } };
    } catch (err) {
      console.error(`[config] Failed to parse config.json: ${err.message}. Using defaults.`);
      _config = { ...DEFAULTS };
    }
  } else {
    _config = { ...DEFAULTS };
  }

  // Auto-generate auth token if not set
  if (!_config.authToken) {
    _config.authToken = randomBytes(24).toString('hex');
    try {
      writeFileSync(CONFIG_PATH, JSON.stringify(_config, null, 2) + '\n', 'utf-8');
      console.log(`[config] Generated auth token. Access UI at: http://localhost:${_config.port}?token=${_config.authToken}`);
    } catch (err) {
      console.error(`[config] Could not save generated auth token: ${err.message}`);
    }
  }

  return _config;
}

export function getConfig() {
  if (!_config) loadConfig();
  return _config;
}

export function updateConfig(updates) {
  const current = getConfig();
  const merged = { ...current, ...updates };
  if (updates.models) {
    merged.models = { ...current.models, ...updates.models };
  }
  writeFileSync(CONFIG_PATH, JSON.stringify(merged, null, 2) + '\n', 'utf-8');
  _config = merged;
  return merged;
}

export function getDbPath() {
  return join(__dirname, 'agent.db');
}

export function getAgentRoot() {
  return __dirname;
}
