import Database from 'better-sqlite3';
import { getDbPath } from './config.mjs';

let db;
let broadcast = () => {}; // set by server.mjs

export function setBroadcast(fn) {
  broadcast = fn;
}

export function initDb() {
  db = new Database(getDbPath());
  db.pragma('journal_mode = WAL');
  db.pragma('foreign_keys = ON');

  db.exec(`
    CREATE TABLE IF NOT EXISTS events (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      type TEXT NOT NULL,
      source TEXT NOT NULL,
      content TEXT NOT NULL,
      metadata TEXT,
      created_at TEXT DEFAULT (datetime('now'))
    );

    CREATE TABLE IF NOT EXISTS memories (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      key TEXT UNIQUE NOT NULL,
      content TEXT NOT NULL,
      category TEXT DEFAULT 'general',
      created_at TEXT DEFAULT (datetime('now')),
      updated_at TEXT DEFAULT (datetime('now'))
    );

    CREATE TABLE IF NOT EXISTS skills (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      name TEXT UNIQUE NOT NULL,
      description TEXT NOT NULL,
      filename TEXT NOT NULL,
      enabled INTEGER DEFAULT 1,
      created_at TEXT DEFAULT (datetime('now'))
    );

    CREATE INDEX IF NOT EXISTS idx_events_type ON events(type);
    CREATE INDEX IF NOT EXISTS idx_events_created ON events(created_at);
    CREATE INDEX IF NOT EXISTS idx_memories_category ON memories(category);

    CREATE TABLE IF NOT EXISTS api_usage (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      date TEXT NOT NULL,
      model TEXT NOT NULL,
      calls INTEGER DEFAULT 0,
      estimated_cost REAL DEFAULT 0,
      created_at TEXT DEFAULT (datetime('now'))
    );
    CREATE UNIQUE INDEX IF NOT EXISTS idx_usage_date_model ON api_usage(date, model);
  `);

  // Migration: add processed column for existing databases
  try {
    db.exec(`ALTER TABLE events ADD COLUMN processed INTEGER DEFAULT 0`);
  } catch {
    // Column already exists
  }
  db.exec(`CREATE INDEX IF NOT EXISTS idx_events_processed ON events(processed) WHERE processed = 0`);

  // Backfill: mark old user chat messages that already have agent replies as processed
  db.exec(`
    UPDATE events SET processed = 1
    WHERE type = 'chat' AND source = 'user' AND processed = 0
    AND EXISTS (
      SELECT 1 FROM events r
      WHERE r.type = 'chat' AND r.source = 'agent'
      AND r.created_at > events.created_at
      AND json_extract(r.metadata, '$.reply_to') = events.id
    )
  `);

  return db;
}

export function getDb() {
  if (!db) throw new Error('Database not initialized — call initDb() first');
  return db;
}

// BigInt-safe JSON serializer (better-sqlite3 returns BigInt for rowids)
function safeStringify(obj) {
  return JSON.stringify(obj, (_, v) => typeof v === 'bigint' ? Number(v) : v);
}

// Events
export function addEvent(type, source, content, metadata = null) {
  const meta = metadata ? safeStringify(metadata) : null;
  const stmt = getDb().prepare(
    'INSERT INTO events (type, source, content, metadata) VALUES (?, ?, ?, ?)'
  );
  const result = stmt.run(type, source, content, meta);
  const event = {
    id: Number(result.lastInsertRowid),
    type, source, content, metadata: meta,
    created_at: new Date().toISOString().replace('T', ' ').slice(0, 19)
  };
  broadcast({ type: 'event', data: event });
  return event;
}

export function getEvents({ type, limit = 50, offset = 0, since } = {}) {
  let sql = 'SELECT * FROM events';
  const params = [];
  const conditions = [];

  if (type) {
    conditions.push('type = ?');
    params.push(type);
  }
  if (since) {
    conditions.push('created_at > ?');
    params.push(since);
  }
  if (conditions.length) sql += ' WHERE ' + conditions.join(' AND ');

  sql += ' ORDER BY created_at DESC LIMIT ? OFFSET ?';
  params.push(limit, offset);

  return getDb().prepare(sql).all(...params);
}

export function getUnprocessedChats() {
  return getDb().prepare(`
    SELECT * FROM events
    WHERE type = 'chat' AND source = 'user' AND processed = 0
    ORDER BY created_at ASC
  `).all();
}

export function markProcessed(eventIds) {
  if (!eventIds.length) return;
  const placeholders = eventIds.map(() => '?').join(',');
  getDb().prepare(`UPDATE events SET processed = 1 WHERE id IN (${placeholders})`).run(...eventIds);
}

// Memories
export function addMemory(key, content, category = 'general') {
  const stmt = getDb().prepare(`
    INSERT INTO memories (key, content, category)
    VALUES (?, ?, ?)
    ON CONFLICT(key) DO UPDATE SET
      content = excluded.content,
      category = excluded.category,
      updated_at = datetime('now')
  `);
  stmt.run(key, content, category);
}

export function getMemory(key) {
  return getDb().prepare('SELECT * FROM memories WHERE key = ?').get(key);
}

export function getAllMemories(category) {
  if (category) {
    return getDb().prepare('SELECT * FROM memories WHERE category = ? ORDER BY updated_at DESC').all(category);
  }
  return getDb().prepare('SELECT * FROM memories ORDER BY updated_at DESC').all();
}

export function deleteMemory(key) {
  getDb().prepare('DELETE FROM memories WHERE key = ?').run(key);
}

// Clear/compact chat
export function deleteEventsUpTo(id) {
  const count = getDb().prepare('SELECT COUNT(*) as c FROM events WHERE id <= ? AND type = ?').get(id, 'chat');
  getDb().prepare('DELETE FROM events WHERE id <= ? AND type = ?').run(id, 'chat');
  return count.c;
}

export function getChatCount() {
  return getDb().prepare('SELECT COUNT(*) as c FROM events WHERE type = ?').get('chat').c;
}

export function getOldestChats(limit) {
  return getDb().prepare(
    'SELECT * FROM events WHERE type = ? ORDER BY created_at ASC LIMIT ?'
  ).all('chat', limit);
}

export function replaceChatsWithSummary(ids, summary) {
  const placeholders = ids.map(() => '?').join(',');
  getDb().prepare(`DELETE FROM events WHERE id IN (${placeholders}) AND type = 'chat'`).run(...ids);
  const meta = JSON.stringify({ compacted: true, original_count: ids.length });
  const stmt = getDb().prepare(
    'INSERT INTO events (type, source, content, metadata) VALUES (?, ?, ?, ?)'
  );
  stmt.run('chat', 'system', summary, meta);
}

// Skills
export function getSkillsFromDb() {
  return getDb().prepare('SELECT * FROM skills ORDER BY name').all();
}

export function addSkillToDb(name, description, filename) {
  const stmt = getDb().prepare(`
    INSERT INTO skills (name, description, filename)
    VALUES (?, ?, ?)
    ON CONFLICT(name) DO UPDATE SET
      description = excluded.description,
      filename = excluded.filename
  `);
  stmt.run(name, description, filename);
}

export function toggleSkill(name, enabled) {
  getDb().prepare('UPDATE skills SET enabled = ? WHERE name = ?').run(enabled ? 1 : 0, name);
}

// API usage tracking
export function recordApiCall(model, estimatedCost = 0) {
  const date = new Date().toISOString().slice(0, 10);
  getDb().prepare(`
    INSERT INTO api_usage (date, model, calls, estimated_cost)
    VALUES (?, ?, 1, ?)
    ON CONFLICT(date, model) DO UPDATE SET
      calls = calls + 1,
      estimated_cost = estimated_cost + ?
  `).run(date, model, estimatedCost, estimatedCost);
}

export function getDailyUsage(date) {
  const d = date || new Date().toISOString().slice(0, 10);
  return getDb().prepare(`
    SELECT model, calls, estimated_cost FROM api_usage WHERE date = ?
  `).all(d);
}

export function getDailyTotalCost(date) {
  const d = date || new Date().toISOString().slice(0, 10);
  const row = getDb().prepare(`
    SELECT COALESCE(SUM(estimated_cost), 0) as total FROM api_usage WHERE date = ?
  `).get(d);
  return row.total;
}
