/**
 * Evolution Engine — Guides the agent's creative growth using a knowledge graph.
 *
 * The agent's creative output follows a Fibonacci spiral pattern:
 * - BRANCH: Explore a frontier domain connected to what it already knows (~5/9)
 * - DEEPEN: Revisit a known domain at higher complexity (~3/9)
 * - LEAP: Try something completely unconnected (~1/9)
 *
 * This prevents repetition and ensures the agent evolves rather than
 * spinning in circles building the same particle simulations.
 */

import { readFile, writeFile } from 'fs/promises';
import { join, dirname } from 'path';
import { fileURLToPath } from 'url';
import { parse, stringify } from 'yaml';

const __dirname = dirname(fileURLToPath(import.meta.url));
const EVOLUTION_PATH = join(__dirname, 'evolution.yaml');

let state = null;

/**
 * Load evolution state from YAML.
 */
export async function loadEvolution() {
  try {
    const raw = await readFile(EVOLUTION_PATH, 'utf-8');
    state = parse(raw);
  } catch {
    state = createDefaultState();
    await saveEvolution();
  }
  return state;
}

/**
 * Save current state back to YAML.
 */
export async function saveEvolution() {
  if (!state) return;
  const raw = stringify(state, { lineWidth: 120 });
  await writeFile(EVOLUTION_PATH, raw, 'utf-8');
}

function createDefaultState() {
  return {
    goal: { active: null, candidates: [], last_asked: null },
    domains: {},
    frontiers: [],
    spiral: {
      last_action: 'branch',
      branch_count: 0,
      deepen_count: 0,
      leap_count: 0,
      total_creations: 0
    }
  };
}

/**
 * Get the current active goal, or null.
 */
export function getGoal() {
  return state?.goal?.active || null;
}

/**
 * Set the agent's active goal.
 */
export async function setGoal(goal) {
  if (!state) await loadEvolution();
  state.goal.active = goal;
  await saveEvolution();
}

/**
 * Add a goal candidate (discovered through reflection).
 */
export async function addGoalCandidate(candidate) {
  if (!state) await loadEvolution();
  if (!state.goal.candidates.includes(candidate)) {
    state.goal.candidates.push(candidate);
    if (state.goal.candidates.length > 5) state.goal.candidates.shift();
    await saveEvolution();
  }
}

/**
 * Decide what kind of creative action to take next using the golden spiral.
 * Returns: { mode: 'branch'|'deepen'|'leap', context: string }
 */
export async function decideNextMove() {
  if (!state) await loadEvolution();

  const { branch_count, deepen_count, leap_count } = state.spiral;
  const total = branch_count + deepen_count + leap_count || 1;

  // Target ratios: 5:3:1 (golden ratio approximation)
  const branchRatio = branch_count / total;
  const deepenRatio = deepen_count / total;
  const leapRatio = leap_count / total;

  let mode;
  // Pick whichever action is most underrepresented
  if (leapRatio < 1 / 9) {
    mode = 'leap';
  } else if (deepenRatio < 3 / 9) {
    mode = 'deepen';
  } else if (branchRatio < 5 / 9) {
    mode = 'branch';
  } else {
    // All balanced — follow the sequence
    const cycle = total % 9;
    if (cycle < 5) mode = 'branch';
    else if (cycle < 8) mode = 'deepen';
    else mode = 'leap';
  }

  return { mode, context: buildContext(mode) };
}

/**
 * Build rich context for the LLM based on the growth mode.
 */
function buildContext(mode) {
  const domains = state.domains || {};
  const frontiers = state.frontiers || [];
  const goal = state.goal?.active;

  // All previously created filenames
  const allWorks = Object.values(domains).flatMap(d => d.works || []);

  // Sort domains by depth (deepest first)
  const domainEntries = Object.entries(domains).sort((a, b) => (b[1].depth || 0) - (a[1].depth || 0));

  let directive = '';

  if (mode === 'branch') {
    // Explore a frontier connected to existing knowledge
    const shuffledFrontiers = [...frontiers].sort(() => Math.random() - 0.5);
    const targetFrontiers = shuffledFrontiers.slice(0, 3);

    // Find which domains connect to these frontiers
    const connectionMap = {};
    for (const frontier of targetFrontiers) {
      connectionMap[frontier] = domainEntries
        .filter(([, d]) => (d.connections || []).includes(frontier))
        .map(([name]) => name);
    }

    directive = `## Growth Mode: BRANCH (explore new territory connected to what you know)

You should explore one of these FRONTIER domains — areas connected to your existing work but not yet explored deeply:

${targetFrontiers.map(f => `- **${f}** (connected to: ${(connectionMap[f] || []).join(', ') || 'general knowledge'})`).join('\n')}

Build something in one of these frontier areas. You're not starting from scratch — use your experience with ${domainEntries.slice(0, 3).map(([n]) => n).join(', ')} as a foundation, but push into genuinely NEW territory.`;

  } else if (mode === 'deepen') {
    // Revisit a domain at higher complexity
    const candidates = domainEntries.filter(([, d]) => (d.depth || 0) < 4);
    const target = candidates.length ? candidates[Math.floor(Math.random() * Math.min(3, candidates.length))] : null;

    if (target) {
      const [name, data] = target;
      directive = `## Growth Mode: DEEPEN (revisit and transcend)

You should revisit **${name}** (current depth: ${data.depth}/4) and create something MORE SOPHISTICATED than your previous work in this area.

Previous works: ${(data.works || []).join(', ')}

DO NOT recreate any of these. Instead, combine this domain with another domain you know (${domainEntries.filter(([n]) => n !== name).slice(0, 3).map(([n]) => n).join(', ')}) to create something that transcends both. Think: what would a DEEPER exploration look like? More interactivity? More mathematical rigor? More artistic expression?`;
    } else {
      directive = `## Growth Mode: DEEPEN
All domains are at maximum depth. Consider branching into new territory instead.`;
    }

  } else {
    // Leap — try something completely new
    directive = `## Growth Mode: LEAP (venture into the unknown)

Create something in a domain you have NEVER explored before. Not a variation of particles, not another fluid sim, not more fractals. Something genuinely new.

Domains to AVOID (you've already explored these thoroughly):
${domainEntries.map(([name, d]) => `- ${name} (depth ${d.depth}, ${(d.works || []).length} works)`).join('\n')}

Think about what's completely missing from your creative portfolio. Music? Storytelling? Data visualization? Social simulation? Language? Architecture? Biology? Consider areas of human creativity you haven't touched.`;
  }

  // Add goal context if present
  const goalContext = goal
    ? `\n## Current Goal\nYour creative direction: **${goal}**\nLet this goal influence what you build — every creation should connect to this purpose, even tangentially.\n`
    : '';

  // Add anti-repetition guard
  const recentWorks = allWorks.slice(-10);
  const antiRepetition = `
## Anti-Repetition Guard
You have created ${allWorks.length} works total. Your most recent: ${recentWorks.join(', ')}.
DO NOT create anything with a similar name or concept to existing works. Check the list above carefully.`;

  return `${directive}${goalContext}${antiRepetition}`;
}

/**
 * Record a new creation and update the knowledge graph.
 * Called after a build-page or write-doc completes.
 */
export async function recordCreation({ filename, mode, domain, connections }) {
  if (!state) await loadEvolution();

  // Update or create domain
  if (!state.domains[domain]) {
    state.domains[domain] = {
      depth: 1,
      works: [],
      last_visited: new Date().toISOString().slice(0, 10),
      connections: connections || []
    };
    // Remove from frontiers if it was there
    state.frontiers = (state.frontiers || []).filter(f => f !== domain);
  } else {
    const d = state.domains[domain];
    d.last_visited = new Date().toISOString().slice(0, 10);
    // Increase depth if revisited with enough works
    if (d.works.length >= 2 && d.depth < 2) d.depth = 2;
    if (d.works.length >= 4 && d.depth < 3) d.depth = 3;
    if (d.works.length >= 7 && d.depth < 4) d.depth = 4;
    // Merge new connections
    for (const c of (connections || [])) {
      if (!d.connections.includes(c)) d.connections.push(c);
    }
  }

  // Add filename to domain works (avoid duplicates)
  const cleanName = filename.replace(/\.html$|\.md$/, '');
  if (!state.domains[domain].works.includes(cleanName)) {
    state.domains[domain].works.push(cleanName);
  }

  // Add new connections as frontiers (if not already a domain)
  for (const c of (connections || [])) {
    if (!state.domains[c] && !(state.frontiers || []).includes(c)) {
      if (!state.frontiers) state.frontiers = [];
      state.frontiers.push(c);
    }
  }

  // Update spiral counters
  if (mode && state.spiral) {
    if (mode === 'branch') state.spiral.branch_count++;
    else if (mode === 'deepen') state.spiral.deepen_count++;
    else if (mode === 'leap') state.spiral.leap_count++;
    state.spiral.total_creations++;
    state.spiral.last_action = mode;
  }

  await saveEvolution();
}

/**
 * Get evolution context for prompts — a compact summary of the growth state.
 */
export async function getEvolutionContext() {
  if (!state) await loadEvolution();

  const domains = state.domains || {};
  const frontiers = state.frontiers || [];
  const goal = state.goal?.active;
  const candidates = state.goal?.candidates || [];
  const spiral = state.spiral || {};

  const domainList = Object.entries(domains)
    .sort((a, b) => (b[1].depth || 0) - (a[1].depth || 0))
    .map(([name, d]) => `- ${name} (depth ${d.depth}, ${(d.works || []).length} works)`)
    .join('\n');

  const allWorks = Object.values(domains).flatMap(d => d.works || []);

  return `## Evolution State
Goal: ${goal || '(none set — consider asking the user or journaling about possible directions)'}
${candidates.length ? `Goal candidates: ${candidates.join(', ')}` : ''}
Total creations: ${spiral.total_creations || allWorks.length}
Growth pattern: ${spiral.branch_count || 0} branches, ${spiral.deepen_count || 0} deepens, ${spiral.leap_count || 0} leaps

### Explored Domains
${domainList || '(none yet)'}

### Frontiers (unexplored but connected)
${frontiers.slice(0, 8).join(', ') || '(none yet)'}

### All Previous Works
${allWorks.join(', ')}`;
}

/**
 * Check if the agent should ask the user about goals.
 * Returns true if no goal is set and it's been a while since last asking.
 */
export function shouldAskAboutGoals() {
  if (!state) return false;
  if (state.goal?.active) return false;

  const lastAsked = state.goal?.last_asked;
  if (!lastAsked) return true;

  // Ask at most once per day
  const hoursSince = (Date.now() - new Date(lastAsked).getTime()) / (1000 * 60 * 60);
  return hoursSince > 24;
}

/**
 * Mark that we asked about goals (so we don't spam).
 */
export async function markGoalAsked() {
  if (!state) await loadEvolution();
  state.goal.last_asked = new Date().toISOString();
  await saveEvolution();
}
