import { readdir, readFile } from 'fs/promises';
import { join, dirname } from 'path';
import { fileURLToPath } from 'url';

const __dirname = dirname(fileURLToPath(import.meta.url));
const INSPIRATION_DIR = join(__dirname, 'inspiration');

/**
 * List all available inspiration files.
 */
export async function listInspirationFiles() {
  try {
    const files = await readdir(INSPIRATION_DIR);
    return files.filter(f => f.endsWith('.html'));
  } catch {
    return [];
  }
}

/**
 * Pick N random inspiration files and return their names.
 */
export async function pickRandom(count = 3) {
  const files = await listInspirationFiles();
  if (!files.length) return [];
  const shuffled = files.sort(() => Math.random() - 0.5);
  return shuffled.slice(0, Math.min(count, files.length));
}

/**
 * Read an inspiration file's content (truncated for prompt use).
 */
export async function readInspirationFile(filename, maxChars = 4000) {
  const filePath = join(INSPIRATION_DIR, filename);
  const content = await readFile(filePath, 'utf-8');
  return content.slice(0, maxChars);
}

/**
 * Get a formatted inspiration block for use in prompts.
 * Picks random files and returns a description of the collection plus a sample.
 */
export async function getInspirationContext({ sampleCount = 1, listCount = 15 } = {}) {
  const all = await listInspirationFiles();
  if (!all.length) return '';

  // Pick a random subset to list
  const listed = all.sort(() => Math.random() - 0.5).slice(0, listCount);
  const names = listed.map(f => f.replace('.html', ''));

  // Pick one to read as a reference
  let sampleContent = '';
  if (sampleCount > 0) {
    const samples = await pickRandom(sampleCount);
    for (const sample of samples) {
      try {
        const content = await readInspirationFile(sample, 3000);
        sampleContent += `\n--- Reference: ${sample} (excerpt) ---\n${content}\n--- end ---\n`;
      } catch {}
    }
  }

  return `## Inspiration Library
You have a collection of ${all.length} interactive HTML pages to draw inspiration from. Some examples:
${names.join(', ')}

These are self-contained HTML files with embedded CSS/JS — games, simulations, art, tools, music, and more. Use them as inspiration for style, structure, and ideas. Don't copy them directly, but let them inform your creative choices.
${sampleContent}`;
}
