import { readFile } from 'fs/promises';
import { join, dirname } from 'path';
import { fileURLToPath, pathToFileURL } from 'url';
import { getConfig } from './config.mjs';
import { recordApiCall, getDailyTotalCost } from './db.mjs';

const __dirname = dirname(fileURLToPath(import.meta.url));

let adapter = null;

const TIER_MAP = {
  health: 'fast',
  format: 'fast',
  classify: 'fast',
  chat: 'balanced',
  journal: 'balanced',
  analyze: 'balanced',
  summarize: 'balanced',
  'self-improve': 'powerful',
  complex: 'powerful',
  build: 'powerful'
};

export async function initLLM() {
  const { cli } = getConfig();

  try {
    const adapterPath = join(__dirname, 'adapters', `${cli}.mjs`);
    adapter = await import(pathToFileURL(adapterPath).href);
  } catch (err) {
    console.error(`[llm] Failed to load adapter "${cli}": ${err.message}`);
    process.exit(1);
  }

  if (!adapter.isAvailable()) {
    console.error(`[llm] CLI "${cli}" is not available on this system.`);
    console.error(`[llm] Install it or change the "cli" field in config.json to an available CLI.`);
    process.exit(1);
  }

  console.log(`[llm] Using ${cli} adapter`);
}

export function pickModel(taskType) {
  const tier = TIER_MAP[taskType] || 'balanced';
  const { models } = getConfig();
  return models[tier] || (adapter ? adapter.modelMap[tier] : tier);
}

// Rough cost estimation per model ($ per 1K chars, ~4 chars per token)
const COST_PER_1K_CHARS = {
  haiku: 0.0001,
  sonnet: 0.0006,
  opus: 0.003,
  default: 0.0006,
};

function estimateCost(prompt, response, model) {
  const modelKey = Object.keys(COST_PER_1K_CHARS).find(k => model?.toLowerCase().includes(k)) || 'default';
  const totalChars = (prompt?.length || 0) + (response?.length || 0);
  return (totalChars / 1000) * COST_PER_1K_CHARS[modelKey];
}

export function checkBudget() {
  const { dailyBudget } = getConfig();
  if (!dailyBudget) return true;
  const spent = getDailyTotalCost();
  return spent < dailyBudget;
}

export async function askLLM(prompt, opts = {}) {
  if (!adapter) throw new Error('LLM not initialized — call initLLM() first');

  const { dailyBudget } = getConfig();
  if (dailyBudget) {
    const spent = getDailyTotalCost();
    if (spent >= dailyBudget) {
      throw new Error(`Daily budget exhausted ($${spent.toFixed(2)} / $${dailyBudget.toFixed(2)})`);
    }
  }

  const response = await adapter.ask(prompt, opts);

  // Record usage
  const model = opts.model || 'unknown';
  const cost = estimateCost(prompt, response, model);
  try { recordApiCall(model, cost); } catch {}

  return response;
}

export async function askLLMWithContext(prompt, contextFiles = [], opts = {}) {
  let context = '';
  for (const filePath of contextFiles) {
    try {
      const content = await readFile(filePath, 'utf-8');
      context += `--- ${filePath} ---\n${content}\n\n`;
    } catch {
      context += `--- ${filePath} --- (could not read)\n\n`;
    }
  }
  const fullPrompt = context ? `${context}\n---\n\n${prompt}` : prompt;
  return askLLM(fullPrompt, opts);
}

export function spawnInteractiveLLM(opts) {
  if (!adapter) throw new Error('LLM not initialized');
  if (!adapter.spawnInteractive) throw new Error(`${adapter.name} adapter does not support interactive mode`);
  return adapter.spawnInteractive(opts);
}
