#!/bin/bash
# Reset agent to a clean state
# Usage: ./reset.sh [--hard]
#
# Default: Clears runtime state (db, journal, output, wake-state, evolution)
# --hard:  Also removes site/, config.json, node_modules/ (full reset)

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
cd "$SCRIPT_DIR"

HARD=false
if [ "$1" = "--hard" ]; then
  HARD=true
fi

echo ""
echo "Resetting agent..."
echo ""

# Stop the agent if running
AGENT_PID=$(pgrep -f "node.*agent.mjs" 2>/dev/null | head -1 || true)
if [ -n "$AGENT_PID" ]; then
  echo "  Stopping agent (PID: $AGENT_PID)..."
  kill "$AGENT_PID" 2>/dev/null || true
  sleep 1
fi

# Remove database (all tables: events, memories, skills, api_usage)
if ls agent.db* 1>/dev/null 2>&1; then
  echo "  Removing database..."
  rm -f agent.db agent.db-shm agent.db-wal
fi

# Clear journal
if [ -f journal.md ]; then
  echo "  Clearing journal..."
  echo "# Journal" > journal.md
fi

# Reset wake state
rm -f wake-state.md
echo "  Cleared wake state"

# Reset evolution state (knowledge graph)
rm -f evolution.yaml
echo "  Cleared evolution state"

# Clear heartbeat
rm -f .heartbeat

# Clear output files (agent-generated creations)
if [ -d output ]; then
  echo "  Clearing output files..."
  rm -rf output/*
fi

# Clear workspace (meta-develop sandbox)
if [ -d workspace ]; then
  echo "  Clearing workspace..."
  find workspace -mindepth 1 -not -path 'workspace/.claude*' -delete 2>/dev/null || true
fi

# Clear uploads
if [ -d uploads ]; then
  echo "  Clearing uploads..."
  rm -rf uploads/*
fi

# Clear generated site
if [ -d site ]; then
  echo "  Removing generated site..."
  rm -rf site
fi

# Clear logs
rm -f agent.log watchdog.log

if [ "$HARD" = true ]; then
  echo ""
  echo "  Hard reset — removing config and dependencies..."
  rm -f config.json
  rm -rf node_modules
  echo "  Run ./setup.sh to reconfigure, or npm install && npm start"
fi

echo ""
echo "Reset complete. Run 'npm start' to start fresh."
echo ""
