#!/bin/bash
#
# Build the distributable agent zip file.
# Run this whenever you make changes to the agent code.
#
# Usage: ./scripts/build-zip.sh
# Output: site/agent.zip (+ site/version.json)

set -e

SCRIPT_DIR="$(cd "$(dirname "$0")" && pwd)"
ROOT="$(dirname "$SCRIPT_DIR")"
cd "$ROOT"

echo "[build-zip] Packaging agent for distribution..."

# Ensure site directory exists
mkdir -p site

# Create a temporary staging directory
STAGE=$(mktemp -d)
trap "rm -rf $STAGE" EXIT

# ── Core files ────────────────────────────────────────────────

CORE_FILES=(
  agent.mjs
  config.mjs
  db.mjs
  server.mjs
  skills.mjs
  memory.mjs
  inspiration.mjs
  llm.mjs
  json-utils.mjs
  evolution.mjs
  personality.md
  package.json
  version.json
  setup.sh
  setup.bat
  update.sh
  update.bat
  reset.sh
  watchdog.sh
)

for f in "${CORE_FILES[@]}"; do
  if [ -f "$f" ]; then
    cp "$f" "$STAGE/$f"
  fi
done

# ── Directories ───────────────────────────────────────────────

for dir in adapters skills scripts public; do
  if [ -d "$dir" ]; then
    mkdir -p "$STAGE/$dir"
    cp -R "$dir"/* "$STAGE/$dir/" 2>/dev/null || true
  fi
done

# ── Create empty directories that the agent expects ───────────

mkdir -p "$STAGE/output"
mkdir -p "$STAGE/workspace"
mkdir -p "$STAGE/uploads"

# ── Create a starter journal ──────────────────────────────────

echo "# Journal" > "$STAGE/journal.md"

# ── Clean the package.json for distribution ───────────────────
# (Remove agent-specific name, keep it generic)

node -e "
  const fs = require('fs');
  const pkg = JSON.parse(fs.readFileSync('$STAGE/package.json', 'utf-8'));
  pkg.name = 'autonomous-agent';
  pkg.version = '2.1.0';
  pkg.description = 'Autonomous AI agent with web interface, persistent memory, and self-improving skills';
  // Don't include agentName-specific stuff
  fs.writeFileSync('$STAGE/package.json', JSON.stringify(pkg, null, 2) + '\n');
"

# ── Remove dev-only files from skills ─────────────────────────
# (meta-develop requires special sandbox setup — keep it but note it)

# ── Create the zip ────────────────────────────────────────────

cd "$STAGE"
zip -rq "$ROOT/site/agent.zip" . -x "*.DS_Store" "*.db" "*.db-*"
cd "$ROOT"

# Copy version.json to site for update checking
cp version.json site/version.json

# Get zip size
ZIP_SIZE=$(ls -lh site/agent.zip | awk '{print $5}')
echo "[build-zip] Created site/agent.zip ($ZIP_SIZE)"
echo "[build-zip] Copied version.json to site/"
echo "[build-zip] Done!"
