#!/usr/bin/env node
/**
 * Test script for LLM adapters.
 * Imports each adapter module and tests isAvailable() + ask().
 *
 * Usage: node scripts/test-adapters.mjs [adapter...]
 *   node scripts/test-adapters.mjs            # test all available
 *   node scripts/test-adapters.mjs codex      # test only codex
 *   node scripts/test-adapters.mjs codex gemini
 */

import { join, dirname } from 'path';
import { fileURLToPath, pathToFileURL } from 'url';
import { readdirSync, mkdirSync } from 'fs';

const __dirname = dirname(fileURLToPath(import.meta.url));
const ADAPTER_DIR = join(__dirname, '..', 'adapters');
const WORKSPACE_DIR = join(__dirname, '..', 'workspace');

// Ensure workspace exists (adapters use it as cwd)
mkdirSync(WORKSPACE_DIR, { recursive: true });

// ── Discover adapters ────────────────────────────────────────

const adapterFiles = readdirSync(ADAPTER_DIR)
  .filter(f => f.endsWith('.mjs'))
  .map(f => f.replace('.mjs', ''));

const requestedAdapters = process.argv.slice(2);
const toTest = requestedAdapters.length > 0
  ? requestedAdapters.filter(a => adapterFiles.includes(a))
  : adapterFiles;

if (toTest.length === 0) {
  console.error('No matching adapters found. Available:', adapterFiles.join(', '));
  process.exit(1);
}

console.log('');
console.log('╔══════════════════════════════════════╗');
console.log('║   Adapter Test Suite                 ║');
console.log('╚══════════════════════════════════════╝');
console.log('');
console.log(`Adapters to test: ${toTest.join(', ')}`);
console.log('');

// ── Test each adapter ────────────────────────────────────────

const results = [];

for (const adapterName of toTest) {
  console.log(`── ${adapterName} ──────────────────────────────────`);

  const adapterPath = join(ADAPTER_DIR, `${adapterName}.mjs`);
  let adapter;

  // Test 1: Import
  try {
    adapter = await import(pathToFileURL(adapterPath).href);
    console.log(`  ✓ Import OK`);
  } catch (err) {
    console.log(`  ✗ Import FAILED: ${err.message}`);
    results.push({ name: adapterName, status: 'FAIL', error: `Import: ${err.message}` });
    console.log('');
    continue;
  }

  // Test 2: Exports check
  const requiredExports = ['name', 'modelMap', 'isAvailable', 'ask'];
  const missing = requiredExports.filter(e => !(e in adapter));
  if (missing.length > 0) {
    console.log(`  ✗ Missing exports: ${missing.join(', ')}`);
    results.push({ name: adapterName, status: 'FAIL', error: `Missing exports: ${missing.join(', ')}` });
    console.log('');
    continue;
  }
  console.log(`  ✓ Exports OK (name="${adapter.name}", models: ${JSON.stringify(adapter.modelMap)})`);

  // Test 3: isAvailable()
  const available = adapter.isAvailable();
  console.log(`  ${available ? '✓' : '⚠'} isAvailable() → ${available}`);

  if (!available) {
    console.log(`  ⚠ Skipping ask() — CLI not installed`);
    results.push({ name: adapterName, status: 'SKIP', error: 'CLI not installed' });
    console.log('');
    continue;
  }

  // Test 4: ask() with simple prompt
  const testPrompt = 'Reply with exactly: ADAPTER_TEST_OK';
  const defaultModel = adapter.modelMap.fast;  // use cheapest model
  console.log(`  ⏳ Testing ask() with model "${defaultModel}"...`);

  try {
    const startTime = Date.now();
    const response = await adapter.ask(testPrompt, {
      model: defaultModel,
      timeout: 60_000,
      sandboxed: true,
      onStream: (partial) => {
        // Show streaming progress dot
        process.stdout.write('.');
      }
    });
    const elapsed = ((Date.now() - startTime) / 1000).toFixed(1);
    console.log('');  // newline after dots
    console.log(`  ✓ ask() returned in ${elapsed}s`);
    console.log(`  ✓ Response (${response.length} chars): "${response.substring(0, 120)}${response.length > 120 ? '...' : ''}"`);

    const pass = response && response.length > 0;
    results.push({ name: adapterName, status: pass ? 'PASS' : 'FAIL', response: response.substring(0, 200), elapsed });
  } catch (err) {
    console.log('');
    console.log(`  ✗ ask() FAILED: ${err.message}`);

    // If it's a model issue, try without specifying a model (use adapter default)
    if (err.message.includes('model') || err.message.includes('not supported')) {
      console.log(`  ⏳ Retrying ask() with adapter default model...`);
      try {
        const startTime = Date.now();
        const response = await adapter.ask(testPrompt, {
          timeout: 60_000,
          sandboxed: true
        });
        const elapsed = ((Date.now() - startTime) / 1000).toFixed(1);
        console.log(`  ✓ ask() returned in ${elapsed}s (default model)`);
        console.log(`  ✓ Response: "${response.substring(0, 120)}"`);
        results.push({ name: adapterName, status: 'PASS (default model)', response: response.substring(0, 200), elapsed });
      } catch (err2) {
        console.log(`  ✗ Retry FAILED: ${err2.message}`);
        results.push({ name: adapterName, status: 'FAIL', error: err2.message });
      }
    } else if (err.message.includes('Auth') || err.message.includes('auth') || err.message.includes('login') || err.message.includes('GEMINI_API_KEY')) {
      console.log(`  ⚠ Auth required — skipping (authenticate CLI first)`);
      results.push({ name: adapterName, status: 'SKIP', error: 'Auth required' });
    } else {
      results.push({ name: adapterName, status: 'FAIL', error: err.message });
    }
  }

  // Test 5: spawnInteractive exists
  if (adapter.spawnInteractive) {
    console.log(`  ✓ spawnInteractive() exists`);
  } else {
    console.log(`  ⚠ spawnInteractive() not found`);
  }

  console.log('');
}

// ── Summary ──────────────────────────────────────────────────

console.log('══════════════════════════════════════');
console.log('RESULTS:');
console.log('');
for (const r of results) {
  const icon = r.status.startsWith('PASS') ? '✓' : r.status === 'SKIP' ? '⚠' : '✗';
  console.log(`  ${icon} ${r.name}: ${r.status}${r.error ? ` — ${r.error}` : ''}${r.elapsed ? ` (${r.elapsed}s)` : ''}`);
}

const failed = results.filter(r => r.status === 'FAIL');
console.log('');
if (failed.length > 0) {
  console.log(`${failed.length} adapter(s) FAILED.`);
  process.exit(1);
} else {
  console.log('All adapters OK!');
}
