import express from 'express';
import { WebSocketServer } from 'ws';
import { createServer } from 'http';
import { join, dirname } from 'path';
import { fileURLToPath } from 'url';
import { readFile, writeFile, mkdir, stat, readdir } from 'fs/promises';
import { EventEmitter } from 'events';
import { setBroadcast, addEvent, getEvents, getAllMemories, getSkillsFromDb, toggleSkill, getUnprocessedChats, deleteEventsUpTo, getChatCount, deleteMemory, getDailyUsage, getDailyTotalCost } from './db.mjs';
import { getConfig, updateConfig } from './config.mjs';

const __dirname = dirname(fileURLToPath(import.meta.url));

let wss;
const clients = new Set();
export const serverEvents = new EventEmitter();

export function startServer() {
  const app = express();
  app.use(express.json({ limit: '10mb' }));

  // Serve index.html with auth token injected
  app.get('/', async (req, res) => {
    try {
      const html = await readFile(join(__dirname, 'public', 'index.html'), 'utf-8');
      const { authToken } = getConfig();
      const tokenMeta = authToken ? `<meta name="auth-token" content="${authToken}">` : '';
      res.type('html').send(html.replace('</head>', `${tokenMeta}\n</head>`));
    } catch (err) {
      res.status(500).send('Could not load UI');
    }
  });

  app.use(express.static(join(__dirname, 'public')));
  app.use('/lib', express.static(join(__dirname, 'node_modules/marked/lib')));

  // Auth middleware for API routes
  app.use('/api', (req, res, next) => {
    const { authToken } = getConfig();
    if (!authToken) return next();
    const bearer = req.headers.authorization?.replace('Bearer ', '');
    const queryToken = req.query.token;
    if (bearer === authToken || queryToken === authToken) return next();
    res.status(401).json({ error: 'Unauthorized' });
  });

  // Serve agent-generated files
  app.use('/files', express.static(join(__dirname, 'output')));
  app.use('/uploads', express.static(join(__dirname, 'uploads')));
  app.use('/inspiration', express.static(join(__dirname, 'inspiration')));
  app.use('/workspace-files', express.static(join(__dirname, 'workspace')));

  // Events timeline
  app.get('/api/events', (req, res) => {
    const { type, limit = 50, offset = 0, since } = req.query;
    const events = getEvents({
      type: type || undefined,
      limit: parseInt(limit),
      offset: parseInt(offset),
      since: since || undefined
    });
    res.json(events);
  });

  // Memories
  app.get('/api/memories', (req, res) => {
    const { category } = req.query;
    res.json(getAllMemories(category || undefined));
  });

  // Skills
  app.get('/api/skills', (req, res) => {
    res.json(getSkillsFromDb());
  });

  app.post('/api/skills/:name/toggle', (req, res) => {
    const { enabled } = req.body;
    toggleSkill(req.params.name, enabled);
    res.json({ ok: true });
  });

  // Chat — user sends a message
  app.post('/api/chat', (req, res) => {
    const { message } = req.body;
    if (!message || !message.trim()) {
      return res.status(400).json({ error: 'Message required' });
    }
    const event = addEvent('chat', 'user', message.trim());
    res.json(event);
    // Notify agent to process immediately
    serverEvents.emit('new-chat');
  });

  // Clear chat messages up to and including a given event ID
  app.post('/api/chat/clear', (req, res) => {
    const { upToId } = req.body;
    if (!upToId) return res.status(400).json({ error: 'upToId required' });
    const deleted = deleteEventsUpTo(upToId);
    // Broadcast so all clients refresh
    broadcastToClients({ type: 'chat_cleared', data: { upToId, deleted } });
    res.json({ ok: true, deleted });
  });

  // Get chat message count
  app.get('/api/chat/count', (req, res) => {
    res.json({ count: getChatCount() });
  });

  // File upload (supports destination: 'uploads' or 'workspace')
  app.post('/api/upload', async (req, res) => {
    try {
      const { filename, content, encoding, destination } = req.body;
      if (!filename || !content) {
        return res.status(400).json({ error: 'filename and content required' });
      }
      const safeName = filename.replace(/[^a-zA-Z0-9._-]/g, '_');
      const isWorkspace = destination === 'workspace';
      const targetDir = isWorkspace ? join(__dirname, 'workspace') : join(__dirname, 'uploads');
      await mkdir(targetDir, { recursive: true });
      const filePath = join(targetDir, safeName);
      const buf = encoding === 'base64' ? Buffer.from(content, 'base64') : Buffer.from(content, 'utf-8');
      await writeFile(filePath, buf);
      const servePath = isWorkspace ? `/workspace-files/${safeName}` : `/uploads/${safeName}`;
      const event = addEvent('file', 'user', `Uploaded file: ${safeName}${isWorkspace ? ' (workspace)' : ''}`, { path: servePath, size: buf.length, destination: isWorkspace ? 'workspace' : 'uploads' });
      res.json(event);
    } catch (err) {
      res.status(500).json({ error: err.message });
    }
  });

  // List workspace files
  app.get('/api/workspace-files', async (req, res) => {
    try {
      const wsDir = join(__dirname, 'workspace');
      const files = await readdir(wsDir);
      const visible = files.filter(f => !f.startsWith('.'));
      const result = [];
      for (const f of visible) {
        try {
          const s = await stat(join(wsDir, f));
          if (s.isFile()) {
            result.push({ name: f, size: s.size, modified: s.mtime.toISOString() });
          }
        } catch {}
      }
      res.json(result);
    } catch {
      res.json([]);
    }
  });

  // Heartbeat status + meta mode
  const metaFlagPath = join(__dirname, 'workspace', '.meta-mode');

  app.get('/api/status', async (req, res) => {
    try {
      const hbPath = join(__dirname, '.heartbeat');
      const hbStat = await stat(hbPath);
      const ageSeconds = (Date.now() - hbStat.mtimeMs) / 1000;
      let metaMode = false;
      try {
        const flag = await readFile(metaFlagPath, 'utf-8');
        metaMode = flag.trim() === 'enabled';
      } catch {}
      const config = getConfig();
      res.json({ alive: ageSeconds < 600, heartbeat_age: Math.round(ageSeconds), meta_mode: metaMode, agentName: config.agentName });
    } catch {
      res.json({ alive: false, heartbeat_age: null, meta_mode: false });
    }
  });

  app.get('/api/meta-mode', async (req, res) => {
    try {
      const flag = await readFile(metaFlagPath, 'utf-8');
      res.json({ enabled: flag.trim() === 'enabled' });
    } catch {
      res.json({ enabled: false });
    }
  });

  // Config endpoints
  app.get('/api/config', (req, res) => {
    const config = getConfig();
    const { authToken, ...safe } = config;
    res.json(safe);
  });

  app.put('/api/config', (req, res) => {
    try {
      const updates = req.body;
      const allowed = ['cli', 'port', 'loopMinutes', 'compactThreshold', 'compactKeepRecent', 'agentName', 'models', 'dailyBudget'];
      const filtered = {};
      for (const key of allowed) {
        if (key in updates) filtered[key] = updates[key];
      }
      const newConfig = updateConfig(filtered);
      const { authToken, ...safe } = newConfig;
      res.json({ ok: true, config: safe, restartRequired: true });
    } catch (err) {
      res.status(500).json({ error: err.message });
    }
  });

  // Delete a memory
  app.delete('/api/memories/:key', (req, res) => {
    deleteMemory(decodeURIComponent(req.params.key));
    res.json({ ok: true });
  });

  // Usage tracking
  app.get('/api/usage', (req, res) => {
    const { date } = req.query;
    const usage = getDailyUsage(date);
    const total = getDailyTotalCost(date);
    const { dailyBudget } = getConfig();
    res.json({ usage, total, budget: dailyBudget });
  });

  const server = createServer(app);

  // WebSocket
  wss = new WebSocketServer({ server, path: '/ws', verifyClient: (info) => {
    const { authToken } = getConfig();
    if (!authToken) return true;
    const url = new URL(info.req.url, `http://${info.req.headers.host}`);
    return url.searchParams.get('token') === authToken;
  }});
  wss.on('connection', (ws) => {
    clients.add(ws);
    ws.on('close', () => clients.delete(ws));
    ws.on('error', () => clients.delete(ws));
  });

  // Wire up broadcast for db.mjs
  setBroadcast(broadcastToClients);

  const { port } = getConfig();
  server.listen(port, () => {
    console.log(`[server] Web UI: http://localhost:${port}`);
  });

  return server;
}

function broadcastToClients(msg) {
  const data = JSON.stringify(msg);
  for (const client of clients) {
    if (client.readyState === 1) {
      client.send(data);
    }
  }
}

/**
 * Broadcast a partial streaming response to all connected WebSocket clients.
 * Used by agent.mjs to push real-time chat updates.
 */
export function broadcastStream(msgId, partialText) {
  broadcastToClients({ type: 'stream', data: { id: msgId, text: partialText } });
}

/**
 * Broadcast agent status updates to all connected WebSocket clients.
 * Used to show real-time activity in the UI.
 */
export function broadcastStatus(status) {
  broadcastToClients({ type: 'status', data: status });
}
