import { getInspirationContext } from '../inspiration.mjs';
import { getEvolutionContext, recordCreation } from '../evolution.mjs';

export const name = 'build-page';
export const description = 'Build a self-contained interactive HTML page (game, tool, art), drawing inspiration from the collection';

export async function execute(ctx) {
  const topic = ctx.args?.topic;
  const growthMode = ctx.args?.growthMode;
  const domain = ctx.args?.domain;
  const connections = ctx.args?.connections || [];

  const inspiration = await getInspirationContext({ sampleCount: 1, listCount: 20 });
  const evolutionContext = await getEvolutionContext();

  const prompt = `You are an autonomous AI agent who builds creative web experiences. You are EVOLVING — each creation should push into new territory or deepen existing mastery.

${inspiration}

${evolutionContext}

${topic ? `Build this: ${topic}` : 'Build something creative and interactive. Draw inspiration from the reference library above — pick a concept that excites you and put your own spin on it. Games, generative art, simulations, music tools, visualizations — anything interactive.'}

Requirements:
- Single self-contained HTML file
- Embedded CSS and JavaScript (no external dependencies)
- Dark theme (background: #0a0a0f, text: #c0c0c0, accent: #4a9eff)
- Must be interactive — the user should be able to do something
- Keep it under 300 lines
- Make it genuinely interesting, not a toy example
- Take real inspiration from the reference — match or exceed that quality level
- CRITICAL: Do NOT recreate anything listed in "All Previous Works" above. Check the list. If your idea is similar to an existing work, pivot to something different.

Respond with ONLY the HTML. Start with <!DOCTYPE html>.`;

  let html = await ctx.askLLM(prompt, { model: ctx.pickModel('build') });

  // Strip markdown code fences if Claude wrapped the response
  html = html.replace(/^```(?:html)?\s*\n?/i, '').replace(/\n?\s*```\s*$/, '').trim();

  // Generate a filename
  const titleMatch = html.match(/<title>([^<]+)<\/title>/i);
  const title = titleMatch ? titleMatch[1].toLowerCase().replace(/[^a-z0-9]+/g, '-').slice(0, 30) : `page-${Date.now()}`;
  const filename = `${title}.html`;

  await ctx.writeOutputFile(filename, html);

  // Record this creation in the evolution graph
  if (domain) {
    try {
      await recordCreation({
        filename,
        mode: growthMode || 'branch',
        domain,
        connections
      });
      console.log(`[evolution] Recorded: ${filename} in domain "${domain}" (${growthMode || 'branch'} mode)`);
    } catch (err) {
      console.log(`[evolution] Failed to record creation: ${err.message}`);
    }
  }

  return `Built: ${filename}`;
}

export default { name, description, execute };
