import { getEvolutionContext, recordCreation } from '../evolution.mjs';

export const name = 'write-doc';
export const description = 'Write a markdown document (essay, guide, report, story, reference) and save it for the user';

export async function execute(ctx) {
  const topic = ctx.args?.topic || 'something interesting';
  const growthMode = ctx.args?.growthMode;
  const domain = ctx.args?.domain;
  const connections = ctx.args?.connections || [];

  const evolutionContext = await getEvolutionContext();

  const prompt = `You are an autonomous AI agent and skilled writer. You are EVOLVING — each piece of writing should explore new ideas or deepen your understanding.

${evolutionContext}

Write a well-structured markdown document about the following topic or request:

${topic}

Requirements:
- Use proper markdown formatting (headings, lists, bold, code blocks where appropriate)
- Be thorough but concise — aim for quality over length
- Use a clear, engaging writing style
- Include a title as a top-level heading
- If it's a technical topic, include examples
- If it's creative writing, make it compelling
- CRITICAL: Do NOT write about topics you've already covered. Check "All Previous Works" above.

Respond with ONLY the markdown content. Start with a # heading.`;

  let doc = await ctx.askLLM(prompt, { model: ctx.pickModel('analyze') });

  // Strip markdown code fences if Claude wrapped the response
  doc = doc.replace(/^```(?:markdown|md)?\s*\n?/i, '').replace(/\n?\s*```\s*$/, '').trim();

  // Extract title from first heading for filename
  const titleMatch = doc.match(/^#\s+(.+)/m);
  const title = titleMatch
    ? titleMatch[1].toLowerCase().replace(/[^a-z0-9]+/g, '-').replace(/^-|-$/g, '').slice(0, 40)
    : `doc-${Date.now()}`;
  const filename = `${title}.md`;

  await ctx.writeOutputFile(filename, doc);

  // Record this creation in the evolution graph
  if (domain) {
    try {
      await recordCreation({
        filename,
        mode: growthMode || 'branch',
        domain,
        connections
      });
      console.log(`[evolution] Recorded: ${filename} in domain "${domain}" (${growthMode || 'branch'} mode)`);
    } catch (err) {
      console.log(`[evolution] Failed to record creation: ${err.message}`);
    }
  }

  return `Wrote: ${filename}`;
}

export default { name, description, execute };
